/**************************************************************************
 * Copyright (C) 2011, ShopSite, Inc.
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 3. Neither the name of ShopSite, Inc. nor the names of its contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **************************************************************************/

#ifdef WIN32
#include <windows.h>
#endif

#include "oauthlib.h"
#include "curl.h"
#include "utils.h"

static FILE *fp = NULL;
static CURL *curl = NULL;
static CURLcode curl_rc;
static CURL_DATA curl_data;
static char curl_buf[CURL_ERROR_SIZE];
static char query[4096];

extern char client_id[];
extern char encrypted_secret_key[];
extern char authorization_code[];
extern char authorization_url[];

extern int transfer_timeout;
extern int connect_timeout;
extern int upload_method;

static void cleanup(void);
extern char *error_message;


/***********************************************************************
 *
 * OAuthUpload
 *
 ***********************************************************************/

int OAuthUpload(char *products, char *pages, char *customers, char *results)
{
  RESOURCE resource;
  char timestamp[16];
  char credentials[128];
  char *encoded_credentials;
  char *signature, *nonce;
  char *secret_key;
  int defer_linking;
  int http_port;
  int bOk;
  int i;

  struct curl_httppost *post = NULL;
  struct curl_httppost *last = NULL;
  char *dbname, *upload_file;
  char *cgiParams, *param, *p;
  char *name, *value;
  int server_file;

  memset(curl_buf, 0, sizeof(curl_buf));
  memset(&curl_data, 0, sizeof(curl_data));

  /* Examine the authorization URL to determine which port to use */
  http_port = memcmp(authorization_url, "https:", 6) == 0? 443 : 80;

  /* Prepare client credentials to be presented to the authorization server */
  oauth_nonce(&nonce);
  sprintf(credentials, "%s:%s", client_id, nonce);

  /* Decrypt the secret key */
  decrypt(encrypted_secret_key, &secret_key);  /* use your own encryption/decryption */

  /* Base64 encode the client credentials and sign the result with the secret key */
  oauth_encode(credentials, &encoded_credentials);
  oauth_signature(encoded_credentials, secret_key, &signature);

  /* Erase the secret key */
  SecureZeroMemory(secret_key, strlen(secret_key));
  mem_free(secret_key);
  
  /* Form the query to request an access token and URL from the authorization server */
  sprintf(query, "grant_type=authorization_code&code=%s&client_credentials=%s&signature=%s",
    authorization_code, encoded_credentials, signature);
  
  mem_free(nonce);
  mem_free(signature);
  mem_free(encoded_credentials);

  /* Initialize curl */
  curl_global_init(CURL_GLOBAL_ALL);
  curl = curl_easy_init();

  curl_easy_setopt(curl, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0);
  curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_SSLv3);

  /* Callback function and error setup */
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, saveData);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&curl_data);
  curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, curl_buf);

  /* Set these as required */
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0);
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0);
  curl_easy_setopt(curl, CURLOPT_TIMEOUT, transfer_timeout);
  curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, connect_timeout);

  /* Make request to the authorization server */
  curl_easy_setopt(curl, CURLOPT_URL, authorization_url);
  curl_easy_setopt(curl, CURLOPT_PORT, http_port);

  /* POST is required, no other method is accepted */
  curl_easy_setopt(curl, CURLOPT_POST, 1);
  curl_easy_setopt(curl, CURLOPT_POSTFIELDS, query);
  curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, strlen(query));

  curl_rc = curl_easy_perform(curl);

  if (curl_rc != 0)
  {
    /* handle curl error - error message should be in curl_buf */
    post_error(curl_buf);
    cleanup();
    return FALSE;
  }

  /* Authorization to access a protected resource is returned in the curl data buffer
   * in JavaScript Object Notation (JSON) format. The oauth_resource_access() routine
   * parses the JSON object notation and populates the resource data structure with
   * the information it extracts. This includes the access_token and its properties
   * along with the resource URLS for uploading, downloading and publishing.
   */
  if (oauth_resource_access(curl_data.buffer, &resource) == FALSE)
  {
    /* an error was returned instead */
    post_error(curl_data.buffer);
    cleanup();
    return FALSE;
  }

  /* If both pages and products are to be uploaded, page and product linking
   * can be deferred until after both uploads have been done. */
  defer_linking = products != NULL && pages != NULL;

  /* Create a file to receive upload results if a file name has been given */
  if (results != NULL)
  {
    if ((fp = fopen(results, "wb")) == NULL)
    {
      /* handle file open error */
      sprintf(query, "Unable to open file [%s] for write.\n%s", results, strerror(errno));
      post_error(query);
      cleanup();
      return FALSE;
    }
  }

  for (i = 0; i < 3; i++)
  {
    switch (i)
    {
      case 0:
        dbname = "products";
        upload_file = products;
        break;

      case 1:
        dbname = "pages";
        upload_file = pages;
        defer_linking = FALSE;
        break;

      case 2:
        dbname = "customers";
        upload_file = customers;
        defer_linking = FALSE;
        upload_method = 2;
        break;
    }

    if (upload_file == NULL)
      continue;

    /* Calculate a nonce and timestamp */
    oauth_nonce(&nonce);
    sprintf(timestamp, "%ld", time(NULL));

    /* Form the query to upload the file */
    sprintf(query, "clientApp=%d&dbname=%s", upload_method, dbname);

    /* Check if the upload file to use is a file on the server */
    if ((server_file = memcmp(upload_file, "server:", 7) == 0))
    {
      upload_file += 7;
      sprintf(query + strlen(query), "&filename=%s", upload_file);
    }

    if (defer_linking)
      strcat(query, "&defer_linking=yes");

    /* Put MAC elements in the request body instead of the Authorization header */
    sprintf(query + strlen(query), "&token=%s&timestamp=%s&nonce=%s", resource.access_token,
      timestamp, nonce);

    /* Decrypt the secret key */
    decrypt(encrypted_secret_key, &secret_key);  /* use your own encryption/decryption */

    /* Create a signed message digest (MAC) */
    bOk = oauth_message_digest(resource.upload1_url, query, secret_key, &signature);

    /* Erase the secret key */
    SecureZeroMemory(secret_key, strlen(secret_key));
    mem_free(secret_key);
    
    if (bOk = FALSE)
    {
      sprintf(query, "Creating message digest failed.  Bad request URL [%s] or missing MAC "
        "elements in the query.", resource.upload1_url);
      post_error(query);
      cleanup();
      return FALSE;
    }

    /* Add the MAC signature to the query */
    sprintf(query + strlen(query), "&signature=%s", signature); 

    mem_free(nonce);
    mem_free(signature);

    if (server_file)
    {
      curl_easy_setopt(curl, CURLOPT_POST, 1);
      curl_easy_setopt(curl, CURLOPT_POSTFIELDS, query);
      curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, strlen(query));
    }
    else  /* file needs to be uploaded */
    {
      /* Build a multipart/formdata HTTP POST */
      cgiParams = strdup(query);
      param = strtok(cgiParams, "&");

      while (param != NULL)
      {
        if ((p = strchr(param, '=')) != NULL)
        {
          name = param;
          value = p + 1;
          *p = '\0';

          curl_formadd(&post, &last, CURLFORM_COPYNAME, name,
            CURLFORM_COPYCONTENTS, value, CURLFORM_END);
        }

        param = strtok(NULL, "&");
      }
  
      mem_free(cgiParams);
  
      curl_easy_setopt(curl, CURLOPT_HTTPPOST, post);
      curl_formadd(&post, &last, CURLFORM_COPYNAME, "UploadFile",
        CURLFORM_CONTENTTYPE, "text/xml", CURLFORM_FILE, upload_file, CURLFORM_END);
    }

    if (upload_method > 1)  /* new method */
    {
      /* Direct the upload results to a filestream if one is opened */
      if (fp != NULL)
      {
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, writeData);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)fp);
      }
    }
    else  /* old method */
    {
      /* Direct the server response to the curl_data buffer to receive CGI parameters for dbmake */
      curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, saveData);
      curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&curl_data);
      reset_curl_data(&curl_data);
    }

    /* Make request to upload the file (and update the database if using the new method) */
    curl_easy_setopt(curl, CURLOPT_URL, resource.upload1_url);
    curl_easy_setopt(curl, CURLOPT_PORT, http_port);

    curl_rc = curl_easy_perform(curl);

    curl_formfree(post);
    post = NULL;
    last = NULL;

    if (curl_rc != 0)
    {
      /* handle curl error */
      post_error(curl_buf);
      cleanup();
      return FALSE;
    }

    if (upload_method == 1)  /* old method */
    {
      /* The curl_data buffer should now have CGI parameters that are to be passed to
       * dbmake to update the database (pages or products) from the uploaded file.
       */
      if (memcmp(curl_data.buffer, "clientApp=1", 11) != 0)
      {
        if (curl_data.length > 0)
        {
          /* something else was returned, possibly an HTTP error */
          fwrite(curl_data.buffer, 1, curl_data.length, fp);
          cleanup();
          return TRUE;
        }
        else
        {
          /* no response at all */
          post_error("No response was received from the server.");
          cleanup();
          return FALSE;
        }
      }

      /* Calculate a nonce and timestamp */
      oauth_nonce(&nonce);
      sprintf(timestamp, "%ld", time(NULL));

      /* Form the query to update the database from the uploaded file */
      strcpy(query, curl_data.buffer);
      reset_curl_data(&curl_data);

      /* Put MAC elements in the request body instead of the Authorization header */
      sprintf(query + strlen(query), "&token=%s&timestamp=%s&nonce=%s", resource.access_token,
        timestamp, nonce);

      /* Decrypt the secret key */
      decrypt(encrypted_secret_key, &secret_key);  /* use your own encryption/decryption */

      /* Create a signed message digest (MAC) */
      bOk = oauth_message_digest(resource.upload2_url, query, secret_key, &signature);

      /* Erase the secret key */
      SecureZeroMemory(secret_key, strlen(secret_key));
      mem_free(secret_key);

      if (bOk == FALSE)
      {
        sprintf(query, "Creating message digest failed.  Bad request URL [%s] or missing MAC "
          "elements in the query.", resource.upload2_url);
        post_error(query);
        cleanup();
        return FALSE;
      }

      /* Add the MAC signature to the query */
      sprintf(query + strlen(query), "&signature=%s", signature); 

      mem_free(nonce);
      mem_free(signature);

      curl_easy_setopt(curl, CURLOPT_POST, 1);
      curl_easy_setopt(curl, CURLOPT_POSTFIELDS, query);
      curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, strlen(query));
    
      /* Direct the upload results to a filestream if one is opened */
      if (fp != NULL)
      {
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, writeData);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)fp);
      }
  
      /* Make request to update the database from the uploaded file */
      curl_easy_setopt(curl, CURLOPT_URL, resource.upload2_url);
      curl_rc = curl_easy_perform(curl);

      if (curl_rc != 0)
      {
        /* handle curl error */
        post_error(curl_buf);
        cleanup();
        return FALSE;
      }
    }
  }
  
  cleanup();
  return TRUE;
}


/***********************************************************************
 *
 * cleanup
 *
 ***********************************************************************/

static void cleanup(void)
{
  if (fp != NULL)
  {
    fclose(fp);
    fp = NULL;
  }

  if (curl != NULL)
  {
    curl_easy_cleanup(curl);
    curl_global_cleanup();
    curl = NULL;
  }

  if (curl_data.buffer != NULL)
    mem_free(curl_data.buffer);
}

