/**************************************************************************
 * Copyright (C) 2011, ShopSite, Inc.
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 3. Neither the name of ShopSite, Inc. nor the names of its contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **************************************************************************/

#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <sys/stat.h>
#endif

#include "oauthlib.h"
#include "curl.h"
#include "utils.h"

/* Order Download Ranges */
#define ALL_ORDERS      1
#define WITHIN_DATES    2
#define WITHIN_NUMBERS  3

/* Search On */
#define SEARCH_NAME  0
#define SEARCH_SKU   1

/* Search Filter */
#define CONTAINS      0
#define BEGINS_WITH   1
#define ENDS_WITH     2

/* Product Search */
extern int search_on;
extern int search_filter;
extern CHAR search_term[];

static FILE *fp = NULL;
static CURL *curl = NULL;
static CURLcode curl_rc;
static CURL_DATA curl_data;
static char curl_buf[CURL_ERROR_SIZE];
static char query[4096];

extern char client_id[];
extern char encrypted_secret_key[];
extern char authorization_code[];
extern char authorization_url[];

extern int download_limit;
extern int transfer_timeout;
extern int connect_timeout;
extern int use_ssl_security;
extern int test_connection;
extern int query_server;

extern char merchant_key[];
extern int use_merchant_key;
extern int include_pay_info;
extern int include_cvv2;

extern char startdate[];
extern char enddate[];
extern int startorder;
extern int endorder;
extern int maxorders;
extern int order_range;

extern char *xml_version;
extern char *error_message;

static void addMerchantKeyToQuery(char *query);
static void cleanup(void);


/***********************************************************************
 *
 * OAuthDownload
 *
 ***********************************************************************/

int OAuthDownload(char *dbname, char *results)
{
  RESOURCE resource;
  char timestamp[16];
  char credentials[128];
  char *encoded_credentials;
  char *signature, *nonce;
  char *secret_key;
  int http_port;
  int bOk;
  
  memset(curl_buf, 0, sizeof(curl_buf));
  memset(&curl_data, 0, sizeof(curl_data));

  /* Examine the authorization URL to determine which port to use */
  http_port = memcmp(authorization_url, "https:", 6) == 0? 443 : 80;

  /* Prepare client credentials to be presented to the authorization server */
  oauth_nonce(&nonce);
  sprintf(credentials, "%s:%s", client_id, nonce);

  /* Decrypt the secret key */
  decrypt(encrypted_secret_key, &secret_key);  /* use your own encryption/decryption */

  /* Base64 encode the client credentials and sign the result with the secret key */
  oauth_encode(credentials, &encoded_credentials);
  oauth_signature(encoded_credentials, secret_key, &signature);

  /* Erase the secret key */
  SecureZeroMemory(secret_key, strlen(secret_key));
  mem_free(secret_key);
  
  /* Form the query to request an access token and URL from the authorization server */
  sprintf(query, "grant_type=authorization_code&code=%s&client_credentials=%s&signature=%s",
    authorization_code, encoded_credentials, signature);
  
  mem_free(nonce);
  mem_free(signature);
  mem_free(encoded_credentials);

  /* Initialize curl */
  curl_global_init(CURL_GLOBAL_ALL);
  curl = curl_easy_init();

  curl_easy_setopt(curl, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0);
  curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_SSLv3);

  /* Callback function and error setup */
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, saveData);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&curl_data);
  curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, curl_buf);

  /* Set these as required */
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0);
  curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0);
  curl_easy_setopt(curl, CURLOPT_TIMEOUT, transfer_timeout);
  curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, connect_timeout);

  /* Make request to the authorization server */
  curl_easy_setopt(curl, CURLOPT_URL, authorization_url);
  curl_easy_setopt(curl, CURLOPT_PORT, http_port);

  /* POST is required, no other method is accepted */
  curl_easy_setopt(curl, CURLOPT_POST, 1);
  curl_easy_setopt(curl, CURLOPT_POSTFIELDS, query);
  curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, -1);

  curl_rc = curl_easy_perform(curl);

  if (curl_rc != 0)
  {
    /* handle curl error - error message should be in curl_buf */
    post_error(curl_buf);
    cleanup();
    return FALSE;
  }

  /* Authorization to access a protected resource is returned in the curl data buffer
   * in JavaScript Object Notation (JSON) format. The oauth_resource_access() routine
   * parses the JSON object notation and populates the resource data structure with
   * the information it extracts. This includes the access_token and its properties
   * along with the resource URLS for uploading, downloading and publishing.
   */
  if (oauth_resource_access(curl_data.buffer, &resource) == FALSE)
  {
    /* an error was returned instead */
    post_error(curl_data.buffer);
    cleanup();
    return FALSE;
  }

  /* If option was selected to query the server, redirect the download request to
   * customdump.cgi, which will download a dump of the web server's configuration
   * settings along with the CGI parameters that were passed.
   */
  if (query_server)
  {
    char *p = strrchr(resource.download_url, '/');
    if (p != NULL)
      strcpy(p + 1, "customdump.cgi");
  }

  /* Calculate a nonce and timestamp */
  oauth_nonce(&nonce);
  sprintf(timestamp, "%ld", time(NULL));

  /* Form the query to test a connection */
  if (strcmp(dbname, "test_connection") == 0)
  {
    strcpy(query, "Test=1");
  }
  else
  {
    /* Form the query to access a database resource */
    sprintf(query, "clientApp=1&dbname=%s&version=%s", dbname, xml_version);
  }

  /* Add query parameters specific to products and pages */
  if (strstr("products pages", dbname) != NULL)
  {
    char *param;

    if (strlen(search_term) > 0)
    {
      switch (search_on)
      {
        case SEARCH_NAME:
          param = "name";
          break;

        case SEARCH_SKU:
          param = "sku";
          break;
      }

      sprintf(query + strlen(query), "&search_on=%s", param);

      switch (search_filter)
      {
        case CONTAINS:
          param = "contains";
          break;

        case BEGINS_WITH:
          param = "begins_with";
          break;

        case ENDS_WITH:
          param = "ends_with";
          break;
      }

      sprintf(query + strlen(query), "&search_filter=%s", param);
      sprintf(query + strlen(query), "&search_term=%s", search_term);
    }

    if (download_limit)
      sprintf(query + strlen(query), "&limit=%d", download_limit);
  }

  /* Add query parameters specific to orders */
  if (strcmp(dbname, "orders") == 0)
  {
    if (order_range == WITHIN_DATES)
    {
      if (strlen(startdate) > 0)
        sprintf(query + strlen(query), "&startdate=%s", startdate);

      if (strlen(enddate) > 0)
        sprintf(query + strlen(query), "&enddate=%s", enddate);
    }
    else
    if (order_range == WITHIN_NUMBERS)
    {
      if (startorder)
        sprintf(query + strlen(query), "&startorder=%d", startorder);

      if (endorder)
        sprintf(query + strlen(query), "&endorder=%d", endorder);
    }

    if (download_limit)
      sprintf(query + strlen(query), "&maxorders=%d", download_limit);

    if (include_pay_info)
    {
      if (include_cvv2)
        sprintf(query + strlen(query), "&pay=1");
      else
        sprintf(query + strlen(query), "&pay=no_cvv");

      if (use_merchant_key)
        addMerchantKeyToQuery(query);
    }
    else
      sprintf(query + strlen(query), "&pay=no_cvv");
  }

  /* Add query parameters specific to customers */
  if (strcmp(dbname, "customers") == 0)
  {
    if (download_limit)
      sprintf(query + strlen(query), "&limit=%d", download_limit);
  }

  /* Put MAC elements in the request body instead of the Authorization header */
  sprintf(query + strlen(query), "&token=%s&timestamp=%s&nonce=%s", resource.access_token,
    timestamp, nonce);

  /* Decrypt the secret key */
  decrypt(encrypted_secret_key, &secret_key);  /* use your own encryption/decryption */

  /* Create a signed message digest (MAC) */
  bOk = oauth_message_digest(resource.download_url, query, secret_key, &signature);

  /* Erase the secret key */
  SecureZeroMemory(secret_key, strlen(secret_key));
  mem_free(secret_key);

  if (bOk == FALSE)
  {
    sprintf(query, "Creating message digest failed.  Bad request URL [%s] or missing MAC "
      "elements in the query.", resource.download_url);
    post_error(query);
    cleanup();
    return FALSE;
  }

  /* Add the MAC signature to the query */
  sprintf(query + strlen(query), "&signature=%s", signature); 

  mem_free(nonce);
  mem_free(signature);

  /* Create file to receive download results */
  if ((fp = fopen(results, "wb")) == NULL)
  {
    /* handle file open error */
    sprintf(query, "Unable to open file [%s] for write.\n%s", results, strerror(errno));
    post_error(query);
    cleanup();
    return FALSE;
  }

  /* Callback function setup */
  curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, writeData);
  curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)fp);

  /* Make access request to the resource server */
  curl_easy_setopt(curl, CURLOPT_URL, resource.download_url);
  curl_easy_setopt(curl, CURLOPT_PORT, http_port);

  /* POST is required, no other method is accepted */
  curl_easy_setopt(curl, CURLOPT_POST, 1);
  curl_easy_setopt(curl, CURLOPT_POSTFIELDS, query);
  curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, -1);

  curl_rc = curl_easy_perform(curl);

  if (curl_rc != 0)
  {
    /* handle curl error */
    post_error(curl_buf);
    cleanup();
    return FALSE;
  }

  cleanup();
  return TRUE;
}


/***********************************************************************
 *
 * addMerchantKeyToQuery
 *
 ***********************************************************************/

static void addMerchantKeyToQuery(char *query)
{
  FILE *fp;
  struct _stat fileinfo;
  char *buffer, *encoded_key;
  int size, nbytes;

  if ((fp = fopen(merchant_key, "rb")) != NULL)
  {
    /* Read the merchant key into memory */
    _fstat(_fileno(fp), &fileinfo);
    size = fileinfo.st_size + 1;
    buffer = mem_alloc(size);
    nbytes = fread(buffer, 1, fileinfo.st_size, fp);
    buffer[nbytes] = '\0';
    fclose(fp);

    /* Base64 encode the key and add it to the query */
    oauth_encode(buffer, &encoded_key);
    sprintf(query + strlen(query), "&dkey=%s", encoded_key);
    mem_free(encoded_key);
    
    /* Erase the merchant key */
    SecureZeroMemory(buffer, size);
    mem_free(buffer);
  }
}


/***********************************************************************
 *
 * cleanup
 *
 ***********************************************************************/

static void cleanup(void)
{
  if (fp != NULL)
  {
    fclose(fp);
    fp = NULL;
  }

  if (curl != NULL)
  {
    curl_easy_cleanup(curl);
    curl_global_cleanup();
    curl = NULL;
  }

  if (curl_data.buffer != NULL)
    mem_free(curl_data.buffer);
}

