/**************************************************************************
 * Copyright (C) 2011, ShopSite, Inc.
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 3. Neither the name of ShopSite, Inc. nor the names of its contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **************************************************************************/

#include <io.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#include <openssl/hmac.h>
#include <openssl/rand.h>
#include <openssl/err.h>

#include "base64.h"
#include "oauthlib.h"

#ifndef TRUE
#define FALSE 0
#define TRUE  1
#endif

typedef struct
{
  char *token;
  char *timestamp;
  char *nonce;
} MAC_ELEMS;

typedef struct
{
  char *host;
  char *port;
  char *path;
} HTTP_ELEMS;

static int egd(void);
static char hex_to_char(char hex[2]);
static char *strlowercase(char *string);
static char *percent_encode(char *string);
static char *percent_decode(char *string);
static char *normalize_http_query(char *query, MAC_ELEMS *);
static int get_http_elements(char *request_url, HTTP_ELEMS *);
static int ascending(const void *elem1, const void *elem2);

static char error_message[1024];


/***********************************************************************
 *
 * oauth_nonce
 *
 ***********************************************************************/

int oauth_nonce(char **result)
{
  unsigned char bytes[32];
  char *string;
  int length;
  int nbytes;
  int i;
  
  static int load_crypto_strings = TRUE;

  length = 8;
  nbytes = length / 2;

  if (!RAND_status() && !egd()) /* entropy gathering device */
  {
    strcpy(error_message, "Insufficient entropy for random number generation.");
    *result = strdup("");
    return FALSE;
  }

  if (!RAND_bytes(bytes, nbytes))
  {
    unsigned long error_code = ERR_get_error();
    int i = 0;

    if (load_crypto_strings)
    {
      ERR_load_crypto_strings();
      load_crypto_strings = FALSE;
    }

    strcpy(error_message, "Random byte generation failed.  ");

    while (error_code != 0)
    {
      char *error_msg = ERR_error_string(error_code, NULL);
      int length1 = strlen(error_message);
      int length2 = strlen(error_msg);

      if (length1 + length2 + 2 < sizeof(error_message))
      {
        if (i > 0)
          strcat(error_message, "; ");

        strcat(error_message, error_msg);
      }
      else
        break;

      error_code = ERR_get_error();
      i++;
    }

    *result = strdup("");
    return FALSE;
  }

  string = mem_alloc(nbytes * 2 + 1);

  for (i = 0; i < nbytes; i++)
    sprintf(&string[i*2], "%02x", bytes[i]);

  *result = string;
  return TRUE;
}


/***********************************************************************
 *
 * oauth_encode
 *
 ***********************************************************************/

void oauth_encode(char *string, char **result)
{
  b64_line_size(0);  /* encode without line breaks */
  *result = b64_encode((unsigned char *)string, strlen(string));
}


/***********************************************************************
 *
 * oauth_decode
 *
 ***********************************************************************/

int oauth_decode(char *string, char **result)
{
  char *buffer;
  int length;
  
  buffer = mem_alloc(strlen(string) + 1);
  length = b64_decode(string, buffer);
  buffer[length] = '\0';

  *result = strdup(buffer);
  mem_free(buffer);

  return length;
}


/***********************************************************************
 *
 * oauth_signature
 *
 ***********************************************************************/

void oauth_signature(char *data, char *key, char **result)
{
  char hash[EVP_MAX_MD_SIZE];
  int datalen, keylen, hashlen;

  b64_line_size(0);  /* encode without line breaks */

  datalen = strlen(data);
  keylen = strlen(key);

  HMAC(EVP_sha1(), key, keylen, data, datalen, hash, &hashlen);
  hash[hashlen] = '\0';

  *result = b64_encode((unsigned char *)hash, hashlen);
}


/***********************************************************************
 *
 * oauth_resource_access
 *
 ***********************************************************************/

int oauth_resource_access(char *authorization, RESOURCE *resource)
{
  char *access_token = NULL;
  char *token_type = NULL;
  char *expires_in = NULL;
  char *download_url = NULL;
  char *upload1_url = NULL;
  char *upload2_url = NULL;
  char *publish_url = NULL;
  char *delim = "{ \",\r\n}";
  char *json, *p, *q, *r, *s, *t, *u, *v;

  json = strdup(authorization);

  p = strstr(json, "\"access_token\"");
  q = strstr(json, "\"token_type\"");
  r = strstr(json, "\"expires_in\"");
  s = strstr(json, "\"download_url\"");
  t = strstr(json, "\"upload1_url\"");
  u = strstr(json, "\"upload2_url\"");
  v = strstr(json, "\"publish_url\"");

  if (p != NULL)
  {
    if ((p = strchr(p, ':')) != NULL)
      access_token = strtok(p + 1, delim);
  }

  if (q != NULL)
  {
    if ((q = strchr(q, ':')) != NULL)
     token_type = strtok(q + 1, delim);
  }

  if (r != NULL)
  {
    if ((r = strchr(r, ':')) != NULL)
      expires_in = strtok(r + 1, delim);
  }

  if (s != NULL)
  {
    if ((s = strchr(s, ':')) != NULL)
      download_url = strtok(s + 1, delim);
  }

  if (t != NULL)
  {
    if ((t = strchr(t, ':')) != NULL)
      upload1_url = strtok(t + 1, delim);
  }

  if (u != NULL)
  {
    if ((u = strchr(u, ':')) != NULL)
      upload2_url = strtok(u + 1, delim);
  }

  if (v != NULL)
  {
    if ((v = strchr(v, ':')) != NULL)
      publish_url = strtok(v + 1, delim);
  }

  resource->access_token = strdup(access_token? access_token : "");
  resource->token_type = strdup(token_type? token_type : "");
  resource->expires_in = atoi(expires_in? expires_in : "");

  resource->download_url = strdup(download_url? download_url : "");
  resource->upload1_url  = strdup(upload1_url? upload1_url : "");
  resource->upload2_url  = strdup(upload2_url? upload2_url : "");
  resource->publish_url  = strdup(publish_url? publish_url : "");

  mem_free(json);

  return strlen(resource->access_token) > 0;
}


/***********************************************************************
 *
 * oauth_message_digest
 *
 ***********************************************************************/

int oauth_message_digest(char *request_url, char *query, char *secret_key, char **result)
{
  MAC_ELEMS mac_elems;
  HTTP_ELEMS http_elems;
  char *request_method = "POST";  /* the only method allowed */
  char *body_hash = "";  /* not calculated */
  char *buffer, *p;
  char *element;
  char *signature;
  int length = 0;
  int i;

  memset(&mac_elems,  0, sizeof(mac_elems));
  memset(&http_elems, 0, sizeof(http_elems));

  if (get_http_elements(request_url, &http_elems) == FALSE)
    return FALSE;

  /* Normalize the name/value pairs in the HTTP query */
  query = normalize_http_query(query, &mac_elems);

  /* Allocate a message buffer for nine components */
  length += strlen(query);
  length += strlen(body_hash);
  length += strlen(request_method);
  length += strlen(http_elems.host);
  length += strlen(http_elems.port);
  length += strlen(http_elems.path);
  length += strlen(mac_elems.token);
  length += strlen(mac_elems.nonce);
  length += strlen(mac_elems.timestamp);
  length += 10; /* nine newlines plus a zero byte */

  buffer = mem_alloc(length);

  /* Assemble the message */
  for (i = 0, p = buffer; i < 9; i++)
  {
    element = NULL;

    switch (i)
    {
      case 0:
        element = mac_elems.token;
        break;

      case 1:
        element = mac_elems.timestamp;
        break;

      case 2:
        element = mac_elems.nonce;
        break;

      case 3:
        element = body_hash;
        break;

      case 4:
        element = request_method;
        break;

      case 5:
        element = http_elems.host;
        break;

      case 6:
        element = http_elems.port;
        break;

      case 7:
        element = http_elems.path;
        break;

      case 8:
        element = query;
        break;
    }

    if (element != NULL)
    {
      sprintf(p, "%s\n", element);
      p += strlen(p);
    }
    else
      break;
  }
  
  if (i == 9)
  {
    /* Create a signed hash with the secret key */
    oauth_signature(buffer, secret_key, &signature);
    *result = signature;
  }

  mem_free(mac_elems.token);
  mem_free(mac_elems.timestamp);
  mem_free(mac_elems.nonce);

  mem_free(http_elems.host);
  mem_free(http_elems.port);
  mem_free(http_elems.path);

  mem_free(buffer);
  mem_free(query);

  return i == 9;
}


/***********************************************************************
 *
 * normalize_http_query
 *
 ***********************************************************************/

static char *normalize_http_query(char *query, MAC_ELEMS *mac_elems)
{
  char *mac_elements;
  char *name, *value;
  char *pair, *p;
  char *buffer;
  char **index;
  int count = 0;
  int length = 0;
  int i;
  
  /* Return the values for these MAC elements */
  mac_elements = "token.....timestamp.nonce";  /* indexed by 10 */

  /* Count the number of name/value pairs */
  if (query != NULL && strlen(query) > 0)
  {
    count = 1;
    pair = query;
    while (pair = strchr(pair, '&'))
    {
      count++;
      pair++;
    }
  }

  if (count == 0)
    return strdup("");

  /* Create an index of the name/value pairs for sorting */
  buffer = mem_alloc(strlen(query) + 1);
  index = mem_alloc(count * sizeof(char *));

  query = strdup(query);
  pair = strtok(query, "&");
  count = 0;

  /* Parse the query string for each name/value pair */
  while (pair != NULL)
  {
    name = pair;
  
    if ((p = strchr(pair, '=')) != NULL)
    {
      value = p + 1;
      *p = '\0';
    }
    else
      value = "";
    
    if ((p = strstr(mac_elements, name)) != NULL)
    {
      /* compute the index of the element name */
      i = (p - mac_elements) / 10;

      switch (i)
      {
        case 0:
          mac_elems->token = strdup(value);
          break;

        case 1:
          mac_elems->timestamp = strdup(value);
          break;

        case 2:
          mac_elems->nonce = strdup(value);
          break;
      }
    }
    else
    {
      /* Normalize name/value encoding */
      percent_decode(name);
      percent_decode(value);
      name = percent_encode(name);
      value = percent_encode(value);

      /* Add name/value pair to the index */
      sprintf(buffer, "%s=%s", name, value);
      index[count++] = strdup(buffer);
      length += strlen(buffer) + 1;  /* include space for newline separator */

      mem_free(name);
      mem_free(value);
    }

    pair = strtok(NULL, "&");
  }

  /* Sort the name/value pairs in ascending byte order */
  qsort((void *)index, count, sizeof(char *), ascending);

  /* Recombine the name/value pairs into one string with newline separators */
  mem_free(buffer);
  buffer = mem_alloc(length + 1);
  p = buffer;

  for (i = 0; i < count; i++)
  {
    sprintf(p, i > 0? "\n%s" : "%s", index[i]);
    mem_free(index[i]);
    p += strlen(p);
  }

  mem_free(index);
  mem_free(query);

  return buffer;
}


/***********************************************************************
 *
 * get_http_elements
 *
 ***********************************************************************/

static int get_http_elements(char *request_url, HTTP_ELEMS *http_elems)
{
  char http[16];
  char url[512];
  char path[512];
  char *hostname;
  char *port;
  char *p;
  int i;
  
  /* Parse the request URL to get the individual HTTP elements:
   * http[s]://hostname[:port][/path][?query]
   */
  memcpy(http, request_url, 8);
  http[8] = '\0';
  strlowercase(http);

  if (memcmp(http, "http://", 7) == 0)
  {
    port = "80";
    i = 7;
  }
  else
  if (memcmp(http, "https://", 8) == 0)
  {
    port = "443";
    i = 8;
  }
  else
    i = 0;

  /* Required minimum URL is http[s]://a.com */
  if (i == 0 || (int)strlen(request_url) < i + 5)
    return FALSE;

  memset(url, 0, sizeof(url));
  memset(path, 0, sizeof(path));

  strncpy(url, request_url, sizeof(url) - 1);

  if ((p = strchr(url, '?')) != NULL)
    *p = '\0';

  p = strstr(url, "//");
  hostname = p + 2;
  *p = '\0';

  if ((p = strchr(hostname, '/')) != NULL)
  {
    strncpy(path, p, sizeof(path) - 1);
    *p = '\0';
  }
  else
    path[0] = '\0';

  if ((p = strchr(hostname, ':')) != NULL)
  {
    port = p + 1;
    *p = '\0';
  }

  http_elems->host = strdup(hostname);
  http_elems->port = strdup(port);
  http_elems->path = strdup(path);

  return TRUE;
}


/***********************************************************************
 *
 * ascending
 *
 * compare routine for sorting name/value pairs
 *
 ***********************************************************************/

static int ascending(const void *elem1, const void *elem2)
{
  char **pair1 = (char **)elem1;
  char **pair2 = (char **)elem2;

  return strcmp(*pair1, *pair2);
}


/***********************************************************************
 *
 * percent_encode (%XX)
 *
 ***********************************************************************/

static char *percent_encode(char *string)
{
  char *buffer;
  char *encoded_string;
  char *s = string;
  char bytes[4];
  int i, c;

  buffer = mem_alloc(3 * strlen(string) + 1);
  buffer[0] = '\0';

  while ((c = *s++) != '\0')
  {
    if (32 <= c && c <= 127)
    {
      if (isalpha(c) || isdigit(c) || strchr("-._~", c))
      {
        bytes[0] = c;
        bytes[1] = '\0';
      }
      else
      {
        i = c % 16;
        bytes[0] = '%';
        bytes[1] = c / 16 + '0';
        bytes[2] = (i > 9)? (i - 10 + 'A') : i + '0';
        bytes[3] = '\0';
      }
      
      strcat(buffer, bytes);
    }
  }

  encoded_string = strdup(buffer);
  mem_free(buffer);

  return encoded_string;
}


/***********************************************************************
 *
 * percent_decode (%XX)
 *
 ***********************************************************************/

static char *percent_decode(char *string)
{
  char hex[2];
  char *p = string;
  char *s = p;

  while (*p != '\0')
  {
    if (*p == '%')
    {
      hex[0] = p[1];
      hex[1] = p[2];
      
      if (isxdigit(hex[0]) && isxdigit(hex[1]))
      {
        *s++ = hex_to_char(hex);
        p += 3;
      }
      else
        *s++ = *p++;
    }
    else
    if (*p == '+')
    {
      *s++ = ' ';
      p++;
    }
    else
      *s++ = *p++;
  }

  *s = '\0';

  return string;
}


/***********************************************************************
 *
 * hex_to_char
 *
 ***********************************************************************/

static char hex_to_char(char hex[2])
{
  int c = 16;
  int i, j;

  i = hex[0] - '0';
  j = toupper(hex[0]) - 'A';
  c *= i < 10? i : j + 10;

  i = hex[1] - '0';
  j = toupper(hex[1]) - 'A';
  c += i < 10? i : j + 10;

  return (char)c;
}


/***********************************************************************
 *
 * strlowercase
 *
 ***********************************************************************/

static char *strlowercase(char *string)
{
  char *p = string;

  while (*p)
  {
    *p++ = tolower(*p);
  }

  return string;
}


/***********************************************************************
 *
 * Entropy Gathering Device 
 *
 ***********************************************************************/

#ifdef WIN32
#define mktemp  _mktemp
#define unlink  _unlink
#endif

void gather_entropy_data(char *buffer, char *path);

static void get_temp_path(char *path, int buflen)
{
  #ifdef WIN32
    GetTempPath(buflen, path);
  #else
    strcpy(path, "/tmp/");
  #endif
}

static void add_entropy(void)
{
  char path[256];
  char *rname, *rfile;
  int rand_count = 0;

  get_temp_path(path, sizeof(path));
  rname = path + strlen(path);

  /* repeat up to 10 times for good randomness */
  while (!RAND_status() && rand_count < 10)
  {
    strcpy(rname, "ssl_random.XXXXXX");
    rfile = mktemp(path);

    RAND_write_file(rfile);
    RAND_load_file(rfile, 1024);

    unlink(rfile);
    rand_count++;
  }
}

#ifdef WIN32

static int egd(void)
{
  char path[MAX_PATH];
  unsigned char buf[16];

  add_entropy();

  if (!RAND_status())
  {
    GetTempPath(sizeof(path), path);
    strcat(path, "egads.dat");
    gather_entropy_data(buf, path);
    RAND_seed(buf, sizeof(buf));
  }

  return RAND_status();
}

#else

static int exists(char *pathname)
{
  struct stat sbuf;
  return stat(pathname, &sbuf) == 0;
}

static int egd(void)
{
  char *pools[] = { "/var/run/egd-pool", "/dev/egd-pool", "/etc/egd-pool",
    "/var/run/egd-pool" };
  
  if (!exists("/dev/random") && !exists("/dev/urandom"))
  {
    int i, n = sizeof(pools) / sizeof(char *);
    for (i = 0; i < n; i++)
    {
      if (RAND_egd(pools[i]) >= 0)
        break;
    }
  }

  add_entropy();

  return RAND_status();
}  

#endif

